import { AlfrescoApi, NodesApi } from "@alfresco/js-api";
import { createObjectCsvWriter } from "csv-writer";
import { log, error } from "./logger.js";

const host = process.env.ALFRESCO_HOST;

const alfrescoJsApi = new AlfrescoApi({
  hostEcm: host,
  authType: "BASIC",
});

const nodesApi = new NodesApi(alfrescoJsApi);

// Login
export async function login(username, password) {
  try {
    await alfrescoJsApi.login(username, password);
    log("LOGIN: Conectado a Alfresco");
  } catch (err) {
    error("Error logging a Alfresco: ", err);
  }
}

// Logout
export async function logout() {
  try {
    await alfrescoJsApi.logout();
    log("LOGOUT: Sesión cerrada exitosamente");
  } catch (err) {
    error("Error logging out: ", err);
  }
}

// Generar índice recursivo y guardarlo en CSV
export const generateCsvIndex = async (nodeId, outputFilePath) => {
  const indexData = [];

  const collectIndexData = async (nodeId, parentPath = "") => {
    try {
      const children = await nodesApi.listNodeChildren(nodeId);
      for (const entry of children.list.entries) {
        const node = entry.entry;
        //const nodePath = parentPath;
        const nodePath = `${parentPath}/${node.name}`;

        indexData.push({
          uuid: node.id,
          name: node.name,
          path: nodePath,
          type: node.isFolder ? "Folder" : "File",
        });

        if (node.isFolder) {
          log(`Procesando carpeta: ${nodePath}`);
          await collectIndexData(node.id, nodePath);
        }
      }
    } catch (err) {
      error(`Error procesando nodo ${nodeId}: ${err.message}`);
    }
  };

  try {
    log(`Generando índice para la carpeta ${nodeId}...`);
    await collectIndexData(nodeId);

    const csvWriter = createObjectCsvWriter({
      path: outputFilePath,
      header: [
        { id: "uuid", title: "UUID" },
        { id: "name", title: "Name" },
        { id: "path", title: "Path" },
        { id: "type", title: "Type" },
      ],
    });

    await csvWriter.writeRecords(indexData);
    log(`Índice generado y guardado en ${outputFilePath}`);
  } catch (err) {
    error(`Error generando el índice en CSV: ${err.message}`);
  }
};